// Copyright (C) 2020  Matthew "strager" Glazar
// See end of file for extended copyright information.

#if !defined(__EMSCRIPTEN__)

#include <cerrno>
#include <cstddef>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <ctime>
#include <exception>
#include <limits.h>
#include <quick-lint-js/assert.h>
#include <quick-lint-js/io/file-path.h>
#include <quick-lint-js/io/temporary-directory.h>
#include <quick-lint-js/port/char8.h>
#include <quick-lint-js/port/have.h>
#include <quick-lint-js/port/unreachable.h>
#include <quick-lint-js/port/warning.h>
#include <quick-lint-js/util/enum.h>
#include <quick-lint-js/util/utf-16.h>
#include <random>
#include <string>
#include <string_view>

#if QLJS_HAVE_DIRENT_H
#include <dirent.h>
#endif

#if QLJS_HAVE_STD_FILESYSTEM
#include <filesystem>
#endif

#if QLJS_HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if QLJS_HAVE_MKDTEMP
#include <sys/stat.h>
#endif

#if QLJS_HAVE_UNISTD_H
#include <unistd.h>
#endif

#if QLJS_HAVE_WINDOWS_H
#include <quick-lint-js/port/windows-error.h>
#include <quick-lint-js/port/windows.h>
#endif

QLJS_WARNING_IGNORE_GCC("-Wsuggest-final-methods")

namespace quick_lint_js {
#if QLJS_HAVE_WINDOWS_H
namespace {
std::string get_temp_dir() {
  std::string path;
  // TODO(strager): Call GetTempPathW instead.
  ::DWORD path_buffer_size = ::GetTempPathA(0, path.data());
  if (path_buffer_size == 0) {
    std::fprintf(stderr, "failed to get path to temporary directory: %s\n",
                 windows_last_error_message().c_str());
    std::abort();
  }
  ::DWORD path_length = path_buffer_size - 1;
  path.resize(path_length);
  ::DWORD rc = ::GetTempPathA(path.size() + 1, path.data());
  QLJS_ALWAYS_ASSERT(rc == path_length);
  QLJS_ASSERT(path.back() == '\\');
  return path;
}
}
#endif

std::string Create_Directory_IO_Error::to_string() const {
  return this->io_error.to_string();
}

#if QLJS_HAVE_MKDTEMP
Result<void, Platform_File_IO_Error> make_unique_directory(std::string &path) {
  path += ".XXXXXX";
  if (!::mkdtemp(path.data())) {
    return failed_result(Platform_File_IO_Error{.error = errno});
  }
  return {};
}
#elif QLJS_HAVE_STD_FILESYSTEM
Result<void, Platform_File_IO_Error> make_unique_directory(std::string &path) {
  std::string_view characters = "abcdefghijklmnopqrstuvwxyz";
  std::uniform_int_distribution<std::size_t> character_index_distribution(
      0, characters.size() - 1);

  std::random_device system_rng;
  std::mt19937 rng(/*seed=*/system_rng());

  for (int attempt = 0; attempt < 100; ++attempt) {
    std::string suffix = ".";
    for (int i = 0; i < 10; ++i) {
      suffix += characters[character_index_distribution(rng)];
    }

    Result<void, Create_Directory_IO_Error> create_result =
        create_directory(path + suffix);
    if (!create_result.ok()) {
      continue;
    }

    path += suffix;
    return {};
  }

  // TODO(strager): Return the proper error code from 'create_result'.
  return failed_result(Platform_File_IO_Error{
      .error = 0,
  });
}
#else
#error "Unsupported platform"
#endif

#if QLJS_HAVE_WINDOWS_H
std::string make_temporary_directory() {
  std::string temp_directory_name = get_temp_dir() + "quick-lint-js";
  auto result = make_unique_directory(temp_directory_name);
  if (!result.ok()) {
    std::fprintf(stderr, "failed to create temporary directory: %s\n",
                 result.error_to_string().c_str());
    std::abort();
  }
  return temp_directory_name;
}
#elif QLJS_HAVE_MKDTEMP
std::string make_temporary_directory() {
  std::string temp_directory_name = "/tmp/quick-lint-js";
  auto result = make_unique_directory(temp_directory_name);
  if (!result.ok()) {
    std::fprintf(stderr, "failed to create temporary directory: %s\n",
                 result.error_to_string().c_str());
    std::abort();
  }
  return temp_directory_name;
}
#elif QLJS_HAVE_STD_FILESYSTEM
std::string make_temporary_directory() {
  std::filesystem::path system_temp_dir_path =
      std::filesystem::temp_directory_path();
  std::string temp_directory_name =
      (system_temp_dir_path / "quick-lint-js").string();
  auto result = make_unique_directory(temp_directory_name);
  if (!result.ok()) {
    std::fprintf(stderr, "failed to create temporary directory: %s\n",
                 result.error_to_string().c_str());
    std::abort();
  }
  return temp_directory_name;
}
#else
#error "Unsupported platform"
#endif

Result<void, Create_Directory_IO_Error> create_directory(
    const std::string &path) {
#if QLJS_HAVE_WINDOWS_H
  std::optional<std::wstring> wpath = mbstring_to_wstring(path.c_str());
  if (!wpath.has_value()) {
    QLJS_UNIMPLEMENTED();
  }
  if (!::CreateDirectoryW(wpath->c_str(), /*lpSecurityAttributes=*/nullptr)) {
    ::DWORD error = ::GetLastError();
    bool directory_existed = false;
    if (error == ERROR_ALREADY_EXISTS) {
      ::DWORD attributes = ::GetFileAttributesW(wpath->c_str());
      if (attributes != INVALID_FILE_ATTRIBUTES) {
        directory_existed = attributes & FILE_ATTRIBUTE_DIRECTORY;
      }
    }
    return failed_result(Create_Directory_IO_Error{
        .io_error =
            Windows_File_IO_Error{
                .error = error,
            },
        .is_directory_already_exists_error = directory_existed,
    });
  }
  return {};
#elif QLJS_HAVE_FCNTL_H
  if (::mkdir(path.c_str(), 0755) != 0) {
    int error = errno;
    bool directory_existed = false;
    if (error == EEXIST) {
      struct ::stat s;
      if (::lstat(path.c_str(), &s) == 0) {
        directory_existed = S_ISDIR(s.st_mode);
      }
    }
    return failed_result(Create_Directory_IO_Error{
        .io_error =
            POSIX_File_IO_Error{
                .error = error,
            },
        .is_directory_already_exists_error = directory_existed,
    });
  }
  return {};
#elif QLJS_HAVE_STD_FILESYSTEM
  std::error_code error;
  if (!std::filesystem::create_directory(to_string8(path), error)) {
    // TODO(strager): Return the proper error code from 'error'.
    return failed_result(create_directory_io_error{
        .io_error =
            Platform_File_IO_Error{
                .error = 0,
            },
    });
  }
  return {};
#else
#error "Unsupported platform"
#endif
}

void create_directory_or_exit(const std::string &path) {
  auto result = create_directory(path);
  if (!result.ok()) {
    std::fprintf(stderr, "error: failed to create directory %s: %s\n",
                 path.c_str(), result.error_to_string().c_str());
    std::terminate();
  }
}

QLJS_WARNING_PUSH
QLJS_WARNING_IGNORE_GCC("-Wformat-nonliteral")
Result<std::string, Platform_File_IO_Error> make_timestamped_directory(
    std::string_view parent_directory, const char *format) {
  std::time_t now = std::time(nullptr);
  std::tm *now_tm = std::localtime(&now);

  std::string directory(parent_directory);
  directory += QLJS_PREFERRED_PATH_DIRECTORY_SEPARATOR;
  std::size_t name_begin_index = directory.size();
  std::size_t name_size = 1;
retry:
  directory.resize(name_begin_index + name_size + 1);
  std::size_t actual_name_size = std::strftime(&directory[name_begin_index],
                                               name_size + 1, format, now_tm);
  if (actual_name_size == 0) {
    name_size *= 2;
    goto retry;
  }
  directory.resize(name_begin_index + actual_name_size);

  auto result = make_unique_directory(directory);
  if (!result.ok()) {
    return result.propagate();
  }
  return directory;
}
QLJS_WARNING_POP

namespace {
template <class Char>
bool is_dot_or_dot_dot(const Char *path) {
  return path[0] == '.' &&
         (path[1] == '\0' || (path[1] == '.' && path[2] == '\0'));
}

#if QLJS_HAVE_WINDOWS_H
Result<void, Platform_File_IO_Error> list_directory_raw(
    const char *directory,
    Temporary_Function_Ref<void(::WIN32_FIND_DATAW &)> visit_entry) {
  std::optional<std::wstring> search_pattern = mbstring_to_wstring(directory);
  if (!search_pattern.has_value()) {
    QLJS_UNIMPLEMENTED();
  }
  *search_pattern += L"\\*";

  ::WIN32_FIND_DATAW entry;
  ::HANDLE finder = ::FindFirstFileW(search_pattern->c_str(), &entry);
  if (finder == INVALID_HANDLE_VALUE) {
    return failed_result(Windows_File_IO_Error{
        .error = ::GetLastError(),
    });
  }
  do {
    visit_entry(entry);
  } while (::FindNextFileW(finder, &entry));

  ::DWORD error = ::GetLastError();
  if (error != ERROR_NO_MORE_FILES) {
    return failed_result(Windows_File_IO_Error{
        .error = error,
    });
  }

  ::FindClose(finder);
  return {};
}
#endif

#if QLJS_HAVE_DIRENT_H
Result<void, Platform_File_IO_Error> list_directory_raw(
    const char *directory,
    Temporary_Function_Ref<void(::dirent *)> visit_entry) {
  ::DIR *d = ::opendir(directory);
  if (d == nullptr) {
    return failed_result(POSIX_File_IO_Error{
        .error = errno,
    });
  }
  for (;;) {
    errno = 0;
    ::dirent *entry = ::readdir(d);
    if (!entry) {
      if (errno != 0) {
        return failed_result(POSIX_File_IO_Error{
            .error = errno,
        });
      }
      break;
    }
    visit_entry(entry);
  }
  ::closedir(d);
  return {};
}
#endif
}

Result<void, Platform_File_IO_Error> list_directory(
    const char *directory,
    Temporary_Function_Ref<void(const char *)> visit_file) {
#if QLJS_HAVE_WINDOWS_H
  return list_directory_raw(directory, [&](::WIN32_FIND_DATAW &entry) -> void {
    // TODO(strager): Reduce allocations.
    std::optional<std::string> entry_name =
        wstring_to_mbstring(entry.cFileName);
    if (!entry_name.has_value()) {
      QLJS_UNIMPLEMENTED();
    }
    if (!is_dot_or_dot_dot(entry_name->c_str())) {
      visit_file(entry_name->c_str());
    }
  });
#elif QLJS_HAVE_DIRENT_H
  return list_directory_raw(directory, [&](::dirent *entry) -> void {
    if (!is_dot_or_dot_dot(entry->d_name)) {
      visit_file(entry->d_name);
    }
  });
#else
#error "Unsupported platform"
#endif
}

Result<void, Platform_File_IO_Error> list_directory(
    const char *directory,
    Temporary_Function_Ref<void(const char *, File_Type_Flags)> visit_file) {
#if QLJS_HAVE_WINDOWS_H
  return list_directory_raw(directory, [&](::WIN32_FIND_DATAW &entry) -> void {
    // TODO(strager): Reduce allocations.
    std::optional<std::string> entry_name =
        wstring_to_mbstring(entry.cFileName);
    if (!entry_name.has_value()) {
      QLJS_UNIMPLEMENTED();
    }
    if (!is_dot_or_dot_dot(entry_name->c_str())) {
      File_Type_Flags flags = File_Type_Flags::none;
      if ((entry.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) ==
          FILE_ATTRIBUTE_DIRECTORY) {
        flags = enum_set_flags(flags, File_Type_Flags::is_directory);
      }
      if ((entry.dwFileAttributes & FILE_ATTRIBUTE_REPARSE_POINT) ==
          FILE_ATTRIBUTE_REPARSE_POINT) {
        flags = enum_set_flags(
            flags, File_Type_Flags::is_symbolic_link_or_reparse_point);
      }
      visit_file(entry_name->c_str(), flags);
    }
  });
#elif QLJS_HAVE_DIRENT_H
  std::string temp_path;
  return list_directory_raw(directory, [&](::dirent *entry) -> void {
    if (is_dot_or_dot_dot(entry->d_name)) {
      return;
    }
    File_Type_Flags flags = File_Type_Flags::none;
    switch (entry->d_type) {
    case DT_UNKNOWN: {
      temp_path.clear();
      temp_path += directory;
      temp_path += QLJS_PREFERRED_PATH_DIRECTORY_SEPARATOR;
      temp_path += entry->d_name;
      struct stat s;
      int lstat_rc = ::lstat(temp_path.c_str(), &s);
      if (lstat_rc == -1) {
        if (errno == ENOENT) {
          return;
        }
      } else {
        if (S_ISDIR(s.st_mode)) {
          flags = enum_set_flags(flags, File_Type_Flags::is_directory);
        }
        if (S_ISLNK(s.st_mode)) {
          flags = enum_set_flags(
              flags, File_Type_Flags::is_symbolic_link_or_reparse_point);
        }
      }
      break;
    }

    case DT_DIR:
      flags = enum_set_flags(flags, File_Type_Flags::is_directory);
      break;

    case DT_LNK:
      flags = enum_set_flags(
          flags, File_Type_Flags::is_symbolic_link_or_reparse_point);
      break;

    default:
      break;
    }
    visit_file(entry->d_name, flags);
  });
#else
#error "Unsupported platform"
#endif
}

void List_Directory_Visitor::visit_directory_pre(const std::string &) {
  // Do nothing by default.
}

void List_Directory_Visitor::visit_directory_post(const std::string &) {
  // Do nothing by default.
}

void list_directory_recursively(const char *directory,
                                List_Directory_Visitor &visitor) {
  struct Finder {
    std::string path;
    List_Directory_Visitor &visitor;

    void recurse(int depth) {
      std::size_t path_length = this->path.size();

      auto visit_child = [&](const char *child_name,
                             File_Type_Flags flags) -> void {
        this->path.resize(path_length);
        this->path += QLJS_PREFERRED_PATH_DIRECTORY_SEPARATOR;
        this->path += child_name;
        if (enum_has_flags(flags, File_Type_Flags::is_directory) &&
            !enum_has_flags(
                flags, File_Type_Flags::is_symbolic_link_or_reparse_point)) {
          this->recurse(depth + 1);
        } else {
          this->visitor.visit_file(path, flags);
        }
      };

      this->path.resize(path_length);
      this->visitor.visit_directory_pre(this->path);

      // TODO(strager): Reduce allocations on Windows. Windows uses wchar_t
      // paths and also needs a "\*" suffix.
      Result<void, Platform_File_IO_Error> list =
          list_directory(this->path.c_str(), std::move(visit_child));
      if (!list.ok()) {
        this->visitor.on_error(list.error(), depth);
      }

      this->path.resize(path_length);
      this->visitor.visit_directory_post(this->path);
    }
  };
  Finder f = {directory, visitor};
  f.recurse(0);
}

Result<std::string, Platform_File_IO_Error> get_current_working_directory() {
  std::string cwd;
  Result<void, Platform_File_IO_Error> r = get_current_working_directory(cwd);
  if (!r.ok()) {
    return r.propagate();
  }
  return cwd;
}

#if QLJS_HAVE_WINDOWS_H
Result<void, Platform_File_IO_Error> get_current_working_directory(
    std::string &out) {
  std::wstring cwd;
  Result<void, Platform_File_IO_Error> r = get_current_working_directory(cwd);
  if (!r.ok()) {
    return r.propagate();
  }
  std::optional<std::string> result = wstring_to_mbstring(cwd);
  if (!result.has_value()) {
    QLJS_UNIMPLEMENTED();
  }
  out = std::move(*result);
  return {};
}

Result<void, Platform_File_IO_Error> get_current_working_directory(
    std::wstring &out) {
  // size includes the null terminator.
  DWORD size = ::GetCurrentDirectoryW(0, nullptr);
  if (size == 0) {
    QLJS_UNIMPLEMENTED();
  }
  out.resize(size - 1);
  // length excludes the null terminator.
  DWORD length = ::GetCurrentDirectoryW(size, out.data());
  if (length == 0) {
    QLJS_UNIMPLEMENTED();
  }
  if (length != size - 1) {
    QLJS_UNIMPLEMENTED();
  }

  return {};
}
#else
Result<void, Platform_File_IO_Error> get_current_working_directory(
    std::string &out) {
  // TODO(strager): Is PATH_MAX sufficient? Do we need to keep growing our
  // buffer?
  out.resize(PATH_MAX);
  if (!::getcwd(out.data(), out.size() + 1)) {
    return failed_result(POSIX_File_IO_Error{errno});
  }
  out.resize(std::strlen(out.c_str()));
  return {};
}
#endif

void set_current_working_directory_or_exit(const char *path) {
#if QLJS_HAVE_STD_FILESYSTEM
  std::filesystem::current_path(path);
#else
  if (::chdir(path) != 0) {
    std::fprintf(stderr, "error: failed to set current directory to %s: %s\n",
                 path, std::strerror(errno));
    std::terminate();
  }
#endif
}
}

#endif

// quick-lint-js finds bugs in JavaScript programs.
// Copyright (C) 2020  Matthew "strager" Glazar
//
// This file is part of quick-lint-js.
//
// quick-lint-js is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// quick-lint-js is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with quick-lint-js.  If not, see <https://www.gnu.org/licenses/>.
