// Copyright (C) 2020  Matthew "strager" Glazar
// See end of file for extended copyright information.

#include <array>
#include <cstdio>
#include <gtest/gtest.h>
#include <quick-lint-js/fe/lex.h>
#include <quick-lint-js/util/cast.h>
#include <sstream>
#include <string>
#include <unicode/uchar.h>
#include <unicode/uversion.h>

namespace quick_lint_js {
namespace {
char32_t max_code_point = U'\U0010ffff';

std::string pretty(char32_t c) {
  std::array<char, 30> buffer;
  int rc = std::snprintf(buffer.data(), buffer.size(), "U+%04llx",
                         narrow_cast<unsigned long long>(c));
  QLJS_ASSERT(rc > 0);
  return std::string(buffer.data(), narrow_cast<std::size_t>(rc));
}

bool icu_data_is_valid() {
  std::uint8_t minimum_unicode_major_version = 17;
  std::uint8_t minimum_unicode_minor_version = 0;

  UVersionInfo version;
  ::u_getUnicodeVersion(version);
  if (version[0] > minimum_unicode_major_version ||
      (version[0] == minimum_unicode_major_version &&
       version[1] >= minimum_unicode_minor_version)) {
    return true;
  }

  static bool did_log_warning = false;
  if (!did_log_warning) {
    std::fprintf(stderr,
                 "warning: The ICU library has data for Unicode version "
                 "%u.%u.%u.%u, which is too old. Upgrade ICU to Unicode "
                 "version %u.%u or newer. Skipping tests...\n",
                 version[0], version[1], version[2], version[3],
                 minimum_unicode_major_version, minimum_unicode_minor_version);
    did_log_warning = true;
  }
  return false;
}

TEST(Test_Lex_Unicode, is_initial_identifier_character) {
  if (!icu_data_is_valid()) {
    GTEST_SKIP();
  }

  for (char32_t c = 0; c <= max_code_point; ++c) {
    bool expected =
        c == U'$' ||  //
        c == U'_' ||  //
        ::u_hasBinaryProperty(narrow_cast<::UChar32>(c), UCHAR_ID_START);
    EXPECT_EQ(Lexer::is_initial_identifier_character(c), expected) << pretty(c);
  }
}

TEST(Test_Lex_Unicode, is_identifier_character) {
  if (!icu_data_is_valid()) {
    GTEST_SKIP();
  }

  for (char32_t c = 0; c <= max_code_point; ++c) {
    bool expected =
        c == U'$' ||       //
        c == U'\u200c' ||  //
        c == U'\u200d' ||
        ::u_hasBinaryProperty(narrow_cast<::UChar32>(c), UCHAR_ID_CONTINUE);
    EXPECT_EQ(
        Lexer::is_identifier_character(c, Lexer::Identifier_Kind::javascript),
        expected)
        << pretty(c);
  }
}
}
}

// quick-lint-js finds bugs in JavaScript programs.
// Copyright (C) 2020  Matthew "strager" Glazar
//
// This file is part of quick-lint-js.
//
// quick-lint-js is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// quick-lint-js is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with quick-lint-js.  If not, see <https://www.gnu.org/licenses/>.
