(() => {
  // public/error-box.mjs
  function showErrorMessageBox(mark, posCursorX, diagnostics) {
    const div = createErrorBox(mark, posCursorX, diagnostics);
    let body = document.querySelector("body");
    body.appendChild(div);
  }
  function createErrorBox(markedElement, posCursorX, diagnostics) {
    let div = document.createElement("div");
    const { bottom } = markedElement.getBoundingClientRect();
    div.setAttribute("id", "error-box");
    div.innerText = diagnostics.map((d) => formatDiagnostic(d)).join("\n");
    div.style.position = "fixed";
    div.style.overflow = "auto";
    div.style.top = `${Math.trunc(bottom)}px`;
    div.style.left = `${posCursorX}px`;
    return div;
  }
  function removeErrorMessageBox() {
    let errorBoxElement = document.querySelector("#error-box");
    if (errorBoxElement !== null) {
      errorBoxElement.remove();
    }
  }
  function showErrorMessage(event, markedElement) {
    removeErrorMessageBox();
    const marks = markedElement.querySelectorAll("mark[data-message]");
    let diagnostics = [];
    let hoveredMarkElement = null;
    for (let mark of marks) {
      if (hasPlayingAnimation(mark)) {
        continue;
      }
      const markRect = mark.getBoundingClientRect();
      if (cursorOverMark(event.clientX, event.clientY, markRect)) {
        diagnostics.push(getDiagnosticFromMark(mark));
        hoveredMarkElement = mark;
      }
    }
    if (hoveredMarkElement !== null) {
      showErrorMessageBox(hoveredMarkElement, event.clientX, diagnostics);
    }
  }
  function hasPlayingAnimation(element) {
    if (typeof element.getAnimations !== "function") {
      return false;
    }
    return element.getAnimations().some((animation) => animation.playState === "running");
  }
  function cursorOverMark(cursorPosX, cursorPosY, markRect) {
    const topDownIn = markRect.bottom >= cursorPosY && cursorPosY >= markRect.top;
    const leftRightIn = cursorPosX >= markRect.left && cursorPosX <= markRect.left + markRect.width;
    return topDownIn && leftRightIn;
  }
  function formatDiagnostic(diagnostic) {
    return diagnostic.code ? `${diagnostic.code} - ${diagnostic.message}` : diagnostic.message;
  }
  function getDiagnosticFromMark(markElement) {
    return {
      message: markElement.attributes["data-message"].value,
      code: markElement.attributes["data-code"]?.value,
      severity: markElement.attributes["data-severity"].value
    };
  }
  function blessErrorMarksWithTooltip(hoveredElement, markedElement) {
    hoveredElement.addEventListener("mousemove", (event) => {
      showErrorMessage(event, markedElement);
    });
    hoveredElement.addEventListener("input", removeErrorMessageBox);
    hoveredElement.addEventListener("click", removeErrorMessageBox);
    hoveredElement.addEventListener("mouseout", removeErrorMessageBox);
  }
  document.addEventListener("DOMContentLoaded", () => {
    const codeInput = document.querySelector("#code-input");
    const codeInputMarks = document.querySelector("#code-input-marks");
    if (codeInput !== null && codeInputMarks !== null) {
      blessErrorMarksWithTooltip(codeInput, codeInputMarks);
    }
    for (let codeBlock of document.querySelectorAll("pre > code.javascript")) {
      blessErrorMarksWithTooltip(codeBlock, codeBlock);
    }
  });
})();
