// Copyright (C) 2020  Matthew "strager" Glazar
// See end of file for extended copyright information.

#include <algorithm>
#include <array>
#include <atomic>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <gmock/gmock.h>
#include <gtest/gtest.h>
#include <quick-lint-js/container/async-byte-queue.h>
#include <quick-lint-js/port/char8.h>
#include <quick-lint-js/port/have.h>
#include <quick-lint-js/port/thread.h>
#include <quick-lint-js/tracking-memory-resource.h>
#include <quick-lint-js/util/algorithm.h>
#include <vector>

namespace quick_lint_js {
namespace {
TEST(Test_Async_Byte_Queue, is_empty_after_construction) {
  Async_Byte_Queue q;

  String8 taken_data = q.take_committed_string8();
  EXPECT_EQ(taken_data, u8"");
}

TEST(Test_Async_Byte_Queue, append_and_commit_one_byte) {
  Async_Byte_Queue q;
  q.append_copy(u8'x');
  q.commit();

  String8 taken_data = q.take_committed_string8();
  EXPECT_EQ(taken_data, u8"x");
}

TEST(Test_Async_Byte_Queue, append_and_commit_one_two_bytes) {
  Async_Byte_Queue q;
  q.append_copy(u8'x');
  q.append_copy(u8'y');
  q.commit();

  String8 taken_data = q.take_committed_string8();
  EXPECT_EQ(taken_data, u8"xy");
}

TEST(Test_Async_Byte_Queue, append_two_bytes_commit_one_byte) {
  Async_Byte_Queue q;
  q.append_copy(u8'x');
  q.commit();
  q.append_copy(u8'y');

  String8 taken_data = q.take_committed_string8();
  EXPECT_EQ(taken_data, u8"x");
}

TEST(Test_Async_Byte_Queue, append_byte_after_take) {
  Async_Byte_Queue q;

  q.append_copy(u8'x');
  q.commit();
  String8 taken_data_1 = q.take_committed_string8();

  q.append_copy(u8'y');
  q.commit();
  String8 taken_data_2 = q.take_committed_string8();
  EXPECT_EQ(taken_data_2, u8"y");
}

#if QLJS_HAVE_THREADS
TEST(Test_Async_Byte_Queue, appended_data_is_readable_by_another_thread) {
  constexpr int write_count = 1000;

  Async_Byte_Queue q;
  String8 expected_data;

  std::atomic<bool> writer_done = false;
  Thread writer_thread([&]() {
    for (int i = 0; i < write_count; ++i) {
      Char8 c = static_cast<Char8>(u8'0' + (i % 10));
      q.append_copy(c);
      q.commit();
      expected_data.push_back(c);
    }
    writer_done.store(true);
  });

  String8 taken_data;
  while (!writer_done.load()) {
    taken_data += q.take_committed_string8();
  }
  taken_data += q.take_committed_string8();
  writer_thread.join();

  EXPECT_EQ(taken_data, expected_data);
}
#endif

TEST(Test_Async_Byte_Queue, append_small_pieces_within_single_chunk) {
  Async_Byte_Queue q;
  ASSERT_LT(4 + 8 + 4, q.default_chunk_size);

  void* piece_0 = q.append(4);
  std::memcpy(piece_0, u8"one ", 4);
  void* piece_1 = q.append(8);
  std::memcpy(piece_1, u8"and two ", 8);
  void* piece_2 = q.append(4);
  std::memcpy(piece_2, u8"thr3", 4);
  q.commit();

  String8 taken_data = q.take_committed_string8();
  EXPECT_EQ(taken_data, u8"one and two thr3");
}

TEST(Test_Async_Byte_Queue, append_small_pieces_within_multiple_chunks) {
  Async_Byte_Queue q;

  static constexpr int piece_size = 3;
  String8 expected_data;
  for (Async_Byte_Queue::Size_Type i = 0; i < q.default_chunk_size * 5;
       i += piece_size) {
    std::array<Char8, piece_size> piece;
    fill(piece, u8'a' + (i % 26));
    std::memcpy(q.append(piece.size()), piece.data(), piece.size());
    expected_data += String8_View(piece.data(), piece.size());
  }
  q.commit();

  String8 taken_data = q.take_committed_string8();
  EXPECT_EQ(taken_data, expected_data);
}

TEST(Test_Async_Byte_Queue, take_multiple_chunks_then_append) {
  Async_Byte_Queue q;

  void* chunk_0 = q.append(q.default_chunk_size);
  std::memset(chunk_0, u8'a', q.default_chunk_size);
  void* chunk_1 = q.append(q.default_chunk_size);
  std::memset(chunk_1, u8'b', q.default_chunk_size);
  q.commit();
  String8 taken_data_01 = q.take_committed_string8();

  void* data_2 = q.append(10);
  std::memcpy(data_2, u8"helloworld", 10);
  q.commit();
  String8 taken_data_2 = q.take_committed_string8();
  EXPECT_EQ(taken_data_2, u8"helloworld");
}

TEST(Test_Async_Byte_Queue, append_multiple_chunks_without_taking) {
  Tracking_Memory_Resource leak_detecting_memory;

  {
    Async_Byte_Queue q(&leak_detecting_memory);
    q.append(q.default_chunk_size);
    q.append(q.default_chunk_size);
  }

  EXPECT_EQ(leak_detecting_memory.alive_bytes(), 0)
      << "async_byte_queue should not have leaked memory";
}

TEST(Test_Async_Byte_Queue, commit_big_write_then_small_writes) {
  Async_Byte_Queue q;
  String8 expected_data;

  Char8* data_0 = static_cast<Char8*>(q.append(q.default_chunk_size));
  std::fill_n(data_0, q.default_chunk_size, u8'a');
  expected_data += String8_View(data_0, q.default_chunk_size);
  q.commit();

  Char8* data_1 = static_cast<Char8*>(q.append(2));
  std::fill_n(data_1, 2, u8'b');
  expected_data += String8_View(data_1, 2);
  q.commit();

  Char8* data_2 = static_cast<Char8*>(q.append(2));
  std::fill_n(data_2, 2, u8'c');
  expected_data += String8_View(data_2, 2);
  q.commit();

  String8 taken_data = q.take_committed_string8();
  EXPECT_EQ(taken_data, expected_data);
}

TEST(Test_Async_Byte_Queue, oversized_chunk) {
  Async_Byte_Queue q;
  String8 expected_data;

  Char8* data_0 = static_cast<Char8*>(q.append(q.default_chunk_size * 3));
  std::fill_n(data_0, q.default_chunk_size * 3, u8'a');
  expected_data += String8_View(data_0, q.default_chunk_size * 3);

  Char8* data_1 = static_cast<Char8*>(q.append(2));
  std::fill_n(data_1, 2, u8'b');
  expected_data += String8_View(data_1, 2);
  q.commit();

  String8 taken_data = q.take_committed_string8();
  EXPECT_EQ(taken_data, expected_data);
}

TEST(Test_Async_Byte_Queue,
     taking_multiple_chunks_keeps_all_chunks_alive_for_finalizer) {
  Async_Byte_Queue q;

  Char8* chunk_0 = static_cast<Char8*>(q.append(q.default_chunk_size));
  std::fill_n(chunk_0, q.default_chunk_size, u8'a');
  Char8* chunk_1 = static_cast<Char8*>(q.append(q.default_chunk_size));
  std::fill_n(chunk_1, q.default_chunk_size, u8'b');
  Char8* chunk_2 = static_cast<Char8*>(q.append(q.default_chunk_size / 4));
  std::fill_n(chunk_2, q.default_chunk_size, u8'c');
  q.commit();

  std::vector<Span<const std::byte>> chunks;
  bool finalize_called = false;
  q.take_committed(
      [&](Span<const std::byte> data) { chunks.push_back(data); },
      [&]() {
        ASSERT_EQ(chunks.size(), 3);

        EXPECT_EQ(chunks[0].size(), q.default_chunk_size);
        EXPECT_EQ(chunks[0].data(), reinterpret_cast<std::byte*>(chunk_0));
        EXPECT_EQ(static_cast<Char8>(chunks[0][0]), u8'a');

        EXPECT_EQ(chunks[1].size(), q.default_chunk_size);
        EXPECT_EQ(chunks[1].data(), reinterpret_cast<std::byte*>(chunk_1));
        EXPECT_EQ(static_cast<Char8>(chunks[1][0]), u8'b');

        EXPECT_EQ(chunks[2].size(), q.default_chunk_size / 4);
        EXPECT_EQ(chunks[2].data(), reinterpret_cast<std::byte*>(chunk_2));
        EXPECT_EQ(static_cast<Char8>(chunks[2][0]), u8'c');

        finalize_called = true;
      });
  EXPECT_TRUE(finalize_called);
}

TEST(Test_Async_Byte_Queue, append_aligned) {
  Async_Byte_Queue q;
  String8 expected_data;

  Char8* data_0 = static_cast<Char8*>(q.append(1));
  data_0[0] = u8'a';

  std::uint64_t* data_1;
  q.append_aligned(sizeof(std::uint64_t) * 2, alignof(std::uint64_t),
                   [&](void* data) -> std::size_t {
                     data_1 = static_cast<std::uint64_t*>(data);
                     data_1[0] = 0x1122334455667788ULL;
                     data_1[1] = 0x99aabbccddeeff00ULL;
                     return sizeof(std::uint64_t) * 2;
                   });

  std::uint64_t* data_2;
  q.append_aligned(sizeof(std::uint64_t), alignof(std::uint64_t),
                   [&](void* data) -> std::size_t {
                     data_2 = static_cast<std::uint64_t*>(data);
                     data_2[0] = 0x1010202030304040ULL;
                     return sizeof(std::uint64_t);
                   });

  EXPECT_EQ(reinterpret_cast<std::uintptr_t>(data_1 + 2),
            reinterpret_cast<std::uintptr_t>(data_2))
      << "data_1 should be in the same chunk as data_2";

  q.commit();

  String8 taken_data = q.take_committed_string8();
  EXPECT_EQ(taken_data.size(), 1 + sizeof(std::uint64_t) * 3);
}

TEST(Test_Async_Byte_Queue, append_aligned_with_rewind) {
  Async_Byte_Queue q;
  String8 expected_data;

  Char8* first_byte = static_cast<Char8*>(q.append(1));
  first_byte[0] = u8'a';

  q.append_aligned(sizeof(std::uint32_t) * 3, alignof(std::uint32_t),
                   [](void* data) -> std::size_t {
                     std::uint32_t* out = static_cast<std::uint32_t*>(data);
                     out[0] = 0x11111111ULL;
                     out[1] = 0x22222222ULL;
                     return sizeof(std::uint32_t) * 2;
                   });

  q.commit();

  String8 taken_data = q.take_committed_string8();
  EXPECT_EQ(taken_data.size(), 1 + sizeof(std::uint32_t) * 2);
  EXPECT_THAT(taken_data,
              ::testing::ElementsAreArray<std::uint8_t>({
                  u8'a',                   // first_byte
                  0x11, 0x11, 0x11, 0x11,  // out[0]
                  0x22, 0x22, 0x22, 0x22,  // out[1]
              }));
}
}
}

// quick-lint-js finds bugs in JavaScript programs.
// Copyright (C) 2020  Matthew "strager" Glazar
//
// This file is part of quick-lint-js.
//
// quick-lint-js is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// quick-lint-js is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with quick-lint-js.  If not, see <https://www.gnu.org/licenses/>.
